# -*- coding: utf-8 -*-

# Exemples de programmes en Python répondant au premier défi, sur la primalité.

import math
import numpy as np


# 1)
# Un test de primalité simple.
# La fonction math.isqrt prend en argument un entier, et renvoie la partie entière de sa racine carrée.
# L'utilisation de la racine carrée permet de tester des entiers jusqu'à environ 10^16.

def test_premier (n) :
    for k in range(2, math.isqrt(n)+1) :
        if n % k == 0 :
            return False
    return True

print("L'entier 2 147 483 647 est premier :", test_premier(2147483647))


# 2)
# Le crible d'Eratosthène

def crible_Eratosthene (n) :
    est_premier = [True for _ in range(n+1)]
    est_premier[0], est_premier[1] = False, False
    for k in range(2, n+1) :
        if est_premier[k] :
            for j in range(2*k, n+1, k) :
                est_premier[j] = False
    return [p for p in range(n+1) if est_premier[p]]

print("Il y a", len(crible_Eratosthene(10**5)), "nombres premiers inférieurs à 100 000.")


# 3)
# Un test de primalité paresseux.
# Rappelons qu'un entier n est premier si et seulement si k**(n-1) = 1 mod n pour tout entier 1 <= k < n.
# En particulier, si n >= 3 et 2**(n-1) != 1 mod n, alors n n'est pas premier. On peut ainsi éliminer beauocup de nombres premiers.
# Le cas n = 2 est à traiter à part.

# Au passage, remarquons que l'expression 'pow(2, n-1, n) == 1' ci-dessous est un booléen : la fonction 'test_primalite_paresseux' 
# retourne True ou False. Cette syntaxe est plus souvent utilisée dans les boucles conditionnelles (if ...).

def test_Fermat_naif (n) :
    if n == 2 :
        return True
    return pow(2, n-1, n) == 1

# Comparons les différences entre cette méthode et le crible d'Eratosthène.
# La méthode naïve introduit des faux positifs (nombres déclarés premiers alors qu'ils sont composites).
# La liste de nombres obtenues est donc plus grande qu'avec le crible d'Eratosthène, qui est exact.

def comparaison_crible (n) :
    premiers_Eratosthene = crible_Eratosthene(n)
    premiers_Fermat_naif = [p for p in range(2, n+1) if test_Fermat_naif(p)]
    return [p for p in premiers_Fermat_naif if p not in premiers_Eratosthene]

print(comparaison_crible(10**3))

# Il y a 3 faux positifs inférieurs ou égaux à 1 000, et 22 faux positifs inférieurs ou égaux à 10 000.
# Ces trois "petits" faux positifs sont : 341 = 11*31, 561 = 3*11*17, 645 = 3*5*43. Remarquons que :
# * phi(341) = 10*30 = 300. L'ordre de 2 dans (Z/341Z)* est 10, qui divise bien 300, mais qui divise aussi 340 (pas de chance !).
# * phi(561) = 2*10*16 = 320. L'ordre de 2 dans (Z/561Z)* est 40, qui divise bien 320, mais qui divise aussi 560 (pas de chance !).
# * phi(645) = 2*4*42 = 336. L'ordre de 2 dans (Z/645Z)* est 28, qui divise bien 336, mais qui divise aussi 644 (pas de chance !).

# Parmis ces 3 "petits" faux positifs, 2 sont des nombres de Carmichael.
# A leur sujet, voir l'article de Wikipedia : https://fr.wikipedia.org/wiki/Nombre_de_Carmichael

# Le test naïf a l'avantage de pourvoir s'appliquer à de très grands nombres, sans avoir à calculer leur factorisation.

p_1 = 422341310396484421031508597853  # Un grand nombre premier (30 chiffres)
p_2 = 977185300400297951195163250231  # Un autre grand nombre premier (30 chiffres)

# print(test_Fermat_naif(p_1), test_Fermat_naif(p_2), test_Fermat_naif(p_1*p_2))


# 4)
# Valuation

def valuation (n, p) :
    if n == 0 :
        return 0 # La valuation p-adique de 0 est un cas particulier.
    k = 0
    while n % p == 0 :
        k += 1
        n = n // p
    return k

M = 89952499728
print(M, "= 3 ^", valuation(M, 3), "*", M // valuation(M, 3))


# 5)
# Facteurs premiers

def crible_facteurs_premiers (n) :
    facteurs_premiers = [[] for n in range(n+1)]
    for k in range(2, n+1) :
        if facteurs_premiers[k] == [] : # C'est-à-dire, si k est premier
            for j in range(k, n+1, k) :
                facteurs_premiers[j].append(k)
    return facteurs_premiers


# Complément
# Pour aller plus loin avec l'algorithme de Fermat naïf :
# Ce test n'est pas si mauvais, mais n'est pas parfait. Pour le reforcer, on peut tester d'autres entiers que 2. Une façon de faire 
# consiste à tester des entiers au hasard. Par exemple, en testant 10 entiers au hasard :

def test_primalite_Fermat (n) :
    for t in range(10) :
        k = np.random.randint(1, n)
        if pow(k, n-1, n) != 1 :
            return False
    return True

def crible_Fermat (N) :
    return [p for p in range(2, N+1) if test_primalite_Fermat(p)]

# Ce test est probabiliste, mais a de bonnes chances de réussir.
# Il y a certains nombres (nombres de Carmichael), plutôt rares, qui se "comportent mal" vis-à-vis de ce test. C'est le cas de 561.
# En-dehors de ces nombres, les chances que le test se trompent sont d'au plus 2**(-10) si l'on teste 10 entiers au hasard.
# En testant plus d'entiers, on peut facilement faire descendre cette probabilité afin d'augmenter la fiabilité du test.
# D'autres tests existent, plus sophistiqués, qui évitent l'écueil des nombres de Camrichael : Miller-Rabin, Lucas, Baillie-PSW...