# -*- coding: utf-8 -*-

# Exemples de programmes en Python répondant au deuxième défi, sur les approximations numériques utilisant des suites.

import numpy as np
import matplotlib.pyplot as plt


# 1)
# Algorithme par dichotomie

# La solution x1(λ) est dans l'intervalle (0,1).
# L'algorithme qui suit peut être simplifié en utilisant la monotonie de la fonction que l'on cherche à annuler, 
# ou en enlevant le compteur.

def f (x, λ) :
    return np.exp(x) - λ*x

def dichotomie_x1 (λ, eps) :
    a, b = 0.0, 1.0
    M = b-a
    k = 0
    while M > eps :
        c = (a+b)/2
        if f(a,λ)*f(c,λ) >= 0 :
            a = c
        else :
            b = c
        M /= 2
        k += 1
    return (a+b)/2, k

print(dichotomie_x1(5, 10**(-12)))

# On calcule x1(5) = 0.259171101819 à 10**(-12) près.
# Il faut pour cela 40 itérations.
# Chaque itération divise l'intervalle par 2. Après k itérations, cet intervalle est donc de taille 2^(-k).
# L'algorithme s'arrête dès que 2^(-k) <= eps, soit k >= -log2 (eps) = -log(eps)/log(2).
# Pour eps = 10^(-12), on trouve k >= 12*log(10)/log(2) ~ 39,9.
# Or k est entier, donc k >= 40.


# 2)
# Algorithme de Newton

# La fonction x -> e^x - λ*x est convexe en x.
# Par conséquent, l'algorithme de Newton démarré en 0 convergera vers la plus petite racine x1 (faites un dessin !).
# L'algorithme qui suit peut être simplifié en enlevant le compteur.

def Newton_x1 (λ, eps) :
    a, b = 0.0, 1.0
    k = 0
    while np.abs(b-a) > eps :
        b = a
        a -= f(a,λ) / (np.exp(a)-λ)
        k += 1
    return a, k

print(Newton_x1(5,10**(-12)))

# On calcule x1(5) = 0.259171101819 à environ 10**(-12) près.
# Il faut pour cela 5 itérations, soit 8 fois moins qu'avec la méthode par dichotomie..
# On observe l'efficacité de l'algorithme de Newton. Le gain apporté sera d'autant plus important 
# que l'on demande une approximation précise.


# 3)
# Approximation de x2

# Il suffit de trouver une majoration de x2(λ).
# Cela peut se faire ou bien en testant f(2**k,λ) jusqu'à tomber sur une valeur positive, 
# ou bien avec un peu de flair : 2*ln(λ) convient.
# L'algorithme par dichotomie commencera alors avec l'intervalle [1, 2*ln(λ)], et l'algorithme 
# de Newton avec 2*ln(λ) (là encore en bénéficiant de la convexité)

def dichotomie_x2 (λ, eps) :
    a, b = 1.0, 2*np.log(λ)
    M = b-a
    while M > eps :
        c = (a+b)/2
        if f(a,λ)*f(c,λ) >= 0 :
            a = c
        else :
            b = c
        M /= 2
    return (a+b)/2

def Newton_x2 (λ, eps) :
    a, b = 2*np.log(λ), 1.0
    while np.abs(b-a) > eps :
        b = a
        a -= f(a,λ) / (np.exp(a)-λ)
    return a

print(dichotomie_x2(5,10**(-12)), Newton_x2(5,10**(-12)))


# 4)
# Accélération de convergence

def Felicette_integrale (N) :
    return 2 * sum([np.sin((2.0*k/N)**2) for k in range(N)]) / N

# Pour obtenir une convergence plus rapide, on évalue Felicette_integrale(2*N)-2*Felicette_integrale (N).
# L'algorithme qui suit vient sans borne d'erreur, et peut encore être accéléré ;
# par exemple, Felicette_integrale(2**k) est calculé deux fois, alors qu'on peut le garder en mémoire.
# Voir aussi la méthode de Romberg.

def Felicette_integrale_rapide (eps) :
    a, b = 0.0, 1.0
    k = 0
    while np.abs(b-a) > eps :
        b = a
        a = 2*Felicette_integrale(2**(k+1)) - Felicette_integrale(2**k)
        k += 1
    return a

# Felicette_integrale(10**6) = 0.8047772461453607 est une approximation de l'intégrale à environ 10**(-6) près.
# Felicette_integrale_rapide(10**(-12)) = 0.804776489344293 est une approximation de l'intégrale à environ 10**(-12) près.
# Ces deux calculs mettent à peu près autant de temps.