# -*- coding: utf-8 -*-

# Exemples de programmes en Python répondant au troisième défi, sur les approximations numériques d'intégrales.

import numpy as np
import matplotlib.pyplot as plt


# 1)
# Différentes méthodes d'intégration numérique

# 1. La première méthode utilise des sommes de Riemann classiques (le point auquel la fonction f est évaluée est à gauche de l'intervalle) :
# ⁽ᵇ⁻ᵃ⁾⁄ₙ ∑ₖ₌₀ⁿ⁻¹ f(a+k⁽ᵇ⁻ᵃ⁾⁄ₙ) converge vers ∫f(t)dt.
# Pour une fonction de classe C¹, la vitesse de convergence est en O(¹⁄ₙ).
def somme_Riemann_gauche (f, a, b, n) :
    pas = float(b-a)/n
    return pas * sum([f(a + k*pas) for k in range(n)])

# 2. La seconde méthode utilise des sommes de Riemann au point milieu (le point auquel la fonction f est évaluée est au milieu de l'intervalle) :
# ⁽ᵇ⁻ᵃ⁾⁄ₙ ∑ₖ₌₀ⁿ⁻¹ f(a+(k+½)⁽ᵇ⁻ᵃ⁾⁄ₙ) converge vers ∫f(t)dt.
# Pour une fonction de classe C², la vitesse de convergence est en O(¹⁄ₙ²).
def somme_Riemann_milieu (f, a, b, n) :
    pas = float(b-a)/n
    return pas * sum([f(a + (k+0.5)*pas) for k in range(n)])

# 3. La troisième méthode utilise la méthode des trapèzes :
# ⁽ᵇ⁻ᵃ⁾⁄ₙ ∑ₖ₌₀ⁿ⁻¹ ½[f(a+k⁽ᵇ⁻ᵃ⁾⁄ₙ)+f(a+(k+1)⁽ᵇ⁻ᵃ⁾⁄ₙ)] converge vers ∫f(t)dt.
# Pour une fonction de classe C², la vitesse de convergence est en O(¹⁄ₙ²).
def somme_trapeze (f, a, b, n) :
    pas = float(b-a)/n
    return pas * sum([(f(a + k*pas) + f(a + (k+1)*pas))/2 for k in range(n)])

# 4. La quatrième méthode est la méthode de Simpson d'ordre 2. 
# On évalue f à la fois au milieu, à gauche et à droite de chaque intervalle, 
# et on pondère les valeurs obtenues pour éliminer tous les termes d'ordre au plus 3.
# ⁽ᵇ⁻ᵃ⁾⁄ₙ ∑ₖ₌₀ⁿ⁻¹ ⅙[f(a+k⁽ᵇ⁻ᵃ⁾⁄ₙ)+4f(a+(k+½)⁽ᵇ⁻ᵃ⁾⁄ₙ)+f(a+(k+1)⁽ᵇ⁻ᵃ⁾⁄ₙ)] converge vers ∫f(t)dt.
# Pour une fonction de classe C⁴, la vitesse de convergence est en O(¹⁄ₙ⁴)
def somme_Simpson (f, a, b, n) :
    pas = float(b-a)/n
    return pas * sum([(f(a + k*pas) + 4*f(a + (k+0.5)*pas) + f(a + (k+1)*pas))/6 for k in range(n)])
# On peut aussi ajuster somme_Riemann_gauche et somme_trapeze

# 5. La cinquième méthode, ajoutée pour compléter ce panorama, est la méthode de Gauss-Legendre d'ordre 2. 
# On évalue f en deux points bien choisis de chaque subdivision, pondérés judicieusement.
# Pour une fonction de classe C⁴, la vitesse de convergence est en O(¹⁄ₙ⁴)
def somme_Gauss_Legendre (f, a, b, n) :
    x1 = (1-1/np.sqrt(3.0))/2
    x2 = (1+1/np.sqrt(3.0))/2
    pas = float(b-a)/n
    return pas * sum([(f(a + (k+x1)*pas) + f(a + (k+x2)*pas))/2 for k in range(n)])

# On teste ces conq méthodes avec l'intégrale de l'énoncé et n = 100.

def f (t) :
    x = 1+t**2
    return np.arctan(np.sqrt(1+x)) / (x * np.sqrt(1+x))

cible = 5 * np.pi**2 / 96
N = 10**2

print(np.abs(somme_Riemann_gauche(f,0,1,N)-cible), 
    np.abs(somme_Riemann_milieu(f,0,1,N)-cible),
    np.abs(somme_trapeze(f,0,1,N)-cible),
    np.abs(somme_Simpson(f,0,1,N)-cible),
    np.abs(somme_Gauss_Legendre(f,0,1,N)-cible))

# On observe les ordre de grandeur annoncés (et même un peu mieux -- les constantes dans les O
# sont ici assez favorables)


# 2)
# Diagramme log--log

# La méthode 1 devrait faire apparaître des erreurs sur une droite de pente -1.
# Les méthodes 2 et 3 devraient faire apparaître des erreurs sur une droite de pente -2.
# Les méthodes 4 et 5 devraient faire apparaître des erreurs sur une droite de pente -4.
# Avec les méthodes 4 et 5, on se retrouve assez vite limité (autour de N = 1000) par la précision machine.

# La fonction suivante prend en entrée un entier p, en ressort le graphique log--log (en base 10) des erreurs de ces trois méthodes, 
# avec n = 2ᵏ pour 0 ≤ k ≤ p.
# Le graphique affiché est un nuage de points.
# Méthode 1 : rouge
# Méthode 2 : orange
# Méthode 3 : jaune
# Méthode 4 : vert
# Méthode 5 : bleu

def graphique_erreurs_log_log (p) :
    I = 5 * np.pi**2 / 96
    liste_erreurs_gauche = [np.log10(abs(somme_Riemann_gauche(f,0,1,2**k)-I)) for k in range(p+1)]
    liste_erreurs_milieu = [np.log10(abs(somme_Riemann_milieu(f,0,1,2**k)-I)) for k in range(p+1)]
    liste_erreurs_trapeze = [np.log10(abs( somme_trapeze(f,0,1,2**k)-I)) for k in range(p+1)]
    liste_erreurs_Simpson = [np.log10(abs(somme_Simpson(f,0,1,2**k)-I)) for k in range(p+1)]
    liste_erreurs_Gauss_Legendre = [np.log10(abs(somme_Gauss_Legendre(f,0,1,2**k)-I)) for k in range(p+1)]
    liste_entiers = [k*np.log10(2) for k in range(p+1)]
    plt.clf() # Nettoie la mémoire de matplotlib de ce qui pourrait traîner.
    plt.gca().set_aspect('equal') # Normalise les axes du graphique, ce qui aide à estimer la pente des droites.
    plt.scatter(liste_entiers, liste_erreurs_gauche, c='red')
    plt.scatter(liste_entiers, liste_erreurs_milieu, c='orange')
    plt.scatter(liste_entiers, liste_erreurs_trapeze, c='yellow')
    plt.scatter(liste_entiers, liste_erreurs_Simpson, c='green')
    plt.scatter(liste_entiers, liste_erreurs_Gauss_Legendre, c='blue')
    plt.show() # Affiche le graphique. En abscisse : log10(n), en ordonnée, log10(erreur commise par l'approximation).

# On calcule les sommes de Riemann jusqu'à k = 12, soit n = 4096.

graphique_erreurs_log_log(12)


# 3)
# Méthode probabiliste

def test_aleatoire () :
    x, y = np.random.uniform(0,1), np.random.uniform(0,np.pi/2)
    return y <= f(x)

def somme_aleatoire (n) :
    return (np.pi/2) * sum([test_aleatoire() for _ in range(n)]) / n

print(abs(somme_aleatoire(10**4)-cible))

# Cette méthode converge presque sûrement vers l'intégrale de f de par la loi forte des grands nombres.
# L'écart à la limite (donc à l'intégrale) est donné par le théorème central limite.
# Cet écart est, en loi, de l'ordre de O(¹⁄ₙ^½).
# Cette méthode a l'avantage d'être extrêmement générale, mais l'inconvénient d'être beaucoup moins efficace 
# que le plus simple algorithme utilisant des sommes de Riemann.


# 4)
# Une fonction non lipschitzienne sur [0,1]

def g (t) :
    return 1 / np.sqrt(1-t)

print(abs(somme_Riemann_gauche(g,0,1,10**4) - 2))

# On observe que l'erreur commise est de l'ordre de O(¹⁄ₙ^½),
# alors que cette erreur est de l'ordre de O(¹⁄ₙ) pour une fonction lipchitzienne.
# Une minoration de l'erreur est ici facile à obtenir. La fonction g étant croissante,
# les sommes de Riemann à gauche sous-estiment l'erreur. L'erreur est au moins celle commise 
# sur la subdivision la plus à droite, qui est exactement de ¹⁄ₙ^½, et au plus l'intégrale de g 
# sur la subdivision la plus à droite, qui est exactement de 2*¹⁄ₙ^½.