function [final_model, final_param, crit]=RPmodsel_hist(X_v,Y_v,collec,law,type,Cov)
% Model selection algorithm for histogram selection by Resampling Penalization
%
% Copyright (C) 2008 Sylvain Arlot
%
%   This program is free software: you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation, either version 3 of the License, or
%   (at your option) any later version.
%
%   This program is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
%
%   You should have received a copy of the GNU General Public License
%   along with this program.  If not, see <http://www.gnu.org/licenses/>
%
% INPUT:
%
% 'X_v' : vector of the X_i
% 'Y_v' : vector of the Y_i
% 'collec' : a cell containing all the histogram models to be compared
%           each element of collec is a vector, whose elements are the possible jump points in the model (including the bounds of X)
%           for instance [0 0.5 1] for the model of regular histograms of dimension 2
% 'law' : string determining the distribution of the weight vector; can take the following values:
%           'Efr'    : Efron(n) weights
%           'Efr(M)' : Efron(M) weights
%           'Poi'    : Poisson(1) weights
%           'Poi(L)' : Poisson(L) weights
%           'Rho'    : Rho(n/2) weights (or (n-1)/2 if n is odd)
%           'Rho(q)' : Rho(q) weights
%           'Loo'    : Rho(n-1) weights
%           'Rad'    : Rad(1/2) weights
%           'Rad(p)' : Rad(p) weights
% 'type' : string determining the method for computing resampling penalties:
%           'exact' for an exact computation (using the explicit formulas obtained in the EJS paper)
%           'MC(B)' for a Monte-Carlo approximation using B realizations of the weight vector
% 'Cov' : real number, equal to the overpenalization constant; put 1 by default
%
% OUTPUT:
% 
% 'final_model' : vector of the boundaries of the selected partition
% 'final_param' : vector of the estimated values on the pieces of the selected partition
% 'crit' : vector of the values of the penalized criterion for each model

% number of data points
n=numel(X_v);

switch type(1:2)
    case 'ex'
% if exact computation, pre-compute quantities that only depend on n
        R1W_v=R1W(n,law);
        R2W_v=R2W(n,law);
    case 'MC'
% if Monte-Carlo approximation, generate the weight matrix
        B=str2double(type(4:(length(type)-1)));
        W_mat=gener_Wmat(n,law,B);
    otherwise
        fprintf('Warning: unrecognized computation method for the Resampling Penalty');
end

nb_model=numel(collec);
m_vect=(1:nb_model);
ERM=cell(size(collec));
Remp=zeros(size(collec));
pen=zeros(size(collec));
%
% for each model, compute the ERM, the empirical risk and the penalty
for i_m=(1:nb_model)
	model=collec{i_m};
switch type(1:2)
    case 'ex'
        [pen(i_m),Remp(i_m),ERM{i_m}] = RP_exact(X_v,Y_v,model,R1W_v,R2W_v);
    case 'MC'
        [pen(i_m),Remp(i_m),ERM{i_m}] = RP_MonteCarlo(X_v,Y_v,model,W_mat);
end
end%for m=(1:nb_model)

% Multiply the penalty by the constant C = Cov*CW 
pen=Cov*C_W(n,law)*pen;

% Compute the penalized criterion for each model
crit=Remp+pen;

% plot(m_vect,crit,'r-');
% xlabel('m');
% ylabel('penalized criterion');

% Selected model
mh_v=m_vect(crit==min(crit));
% If there are several minimizers, choose the one for which the penalty is
% the smallest
if numel(mh_v)>1
    penmh=pen(mh_v);
    mh=min(mh_v(penmh==min(penmh)));
else
    mh=mh_v;
end%if prod(size(mh_v))>1

 
% Final estimator
final_model=collec{mh};
final_param=ERM{mh};


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%% Useful subfunctions %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%% Compute R_{1,W}, R_{2,W} and C_W:

function R_v=R1W(n,law)
% computes the constant R_{1,W}(n,phl) for every n*phl=1, ..., n
% for a resampling distribution given by 'law'
% possible values for the string law:
% 'Efr'    : Efron(n) weights
% 'Efr(M)' : Efron(M) weights
% 'Poi'    : Poisson(1) weights
% 'Poi(L)' : Poisson(L) weights
% 'Rho'    : Rho(n/2) weights (or (n-1)/2 if n is odd)
% 'Rho(q)' : Rho(q) weights
% 'Loo'    : Rho(n-1) weights
% 'Rad'    : Rad(1/2) weights
% 'Rad(p)' : Rad(p) weights

switch (law(1:3))
    case 'Efr'
        if length(law)==3
            param=n;
        else
            param=str2double(law(5:(length(law)-1)));
        end

        R_v = (n/param)*(1-(1./(1:n))).*Einv_efr(n,param);

    case 'Poi'
        if length(law)==3
            param=1;
        else
            param=str2double(law(5:(length(law)-1)));
        end

        R_v = (1/param)*(1-(1./(1:n))).*Einv_poi(n,param);

    case 'Rho'
        if length(law)==3
            param=floor(n/2);
        else
            param=min(n,str2double(law(5:(length(law)-1))));
        end

        R_v = (((n/param)*(Einv_rho(n,param))) - 1).*ones(1,n);

    case 'Loo'
        R_v = [0 ((2:n)./(n*(1:(n-1))))];

    case 'Rad'
        if length(law)==3
            param=1/2;
		else
            param=str2double(law(5:(length(law)-1)));
       	end

    	if (param<1 && param>0)
    		R_v = (((1/param)*(Einv_rad(n,param)))-1).*ones(1,n);
    	else
    	    R_v = NaN*ones(1,n);
    	end

    otherwise
	R_v = NaN*ones(1,n);

end%switch (law(1:3))

function v=Einv_efr(n,M)
% computes (r*M/n)*E[1/B(M,r/n) | .>0] 
% for r=1..n
v=zeros(1,n);
for i=1:n
    p=i/n;
    v(i)=sum(M*p*(binopdf(1:M,M,p)./(1:M))/(binocdf(M-1,M,1-p)));
end
v(n)=1;

function v=Einv_rad(n,p)
% computes (r*p)*E[1/B(r,p) | .>0] 
% for r=1..n
v=zeros(1,n);
for i=1:n
    v(i)=sum(i*p*(binopdf(1:i,i,p)./(1:i))/(binocdf(i-1,i,1-p)));
end

function v=Einv_rho(n,q)
% computes (r*q/n)*E[1/H(n,r,q) | .>0] 
% for r=1..n
v=zeros(1,n);
for i=1:n
    v(i)=(i*q/n)*sum((hygepdf(1:min(i,q),n,i,q)./(1:min(i,q)))/(1-hygepdf(0,n,i,q)));
end

function v=Einv_poi(n,lambda)
% computes (r*lambda)*E[1/P(r*lambda) | .>0] 
% for r=1..n
[T,Y]=ode45(@ode_fonc_Einv_poisson,lambda*(1:n),0);
v=Y';

function dy=ode_fonc_Einv_poisson(t,y)
dy = 1 - y*(1 - 1/t + (1/(exp(t)-1)));



function R_v=R2W(n,law)
% computes the constant R_{2,W}(n,phl) for every n*phl=1, ..., n
% for a resampling distribution given by 'law'
% possible values for the string law:
% 'Efr'    : Efron(n) weights
% 'Efr(M)' : Efron(M) weights
% 'Poi'    : Poisson(1) weights
% 'Poi(L)' : Poisson(L) weights
% 'Rho'    : Rho(n/2) weights (or (n-1)/2 if n is odd)
% 'Rho(q)' : Rho(q) weights
% 'Loo'    : Rho(n-1) weights
% 'Rad'    : Rad(1/2) weights
% 'Rad(p)' : Rad(p) weights

switch (law(1:3))
    case 'Efr'
        if length(law)==3
            param=n;
        else
            param=str2double(law(5:(length(law)-1)));
        end

        R_v = (n/param)*(1-(1./(1:n)));

    case 'Poi'
        if length(law)==3
            param=1;
        else
            param=str2double(law(5:(length(law)-1)));
        end

        R_v = (1/param)*(1-(1./(1:n)));

    case 'Rho'
        if length(law)==3
            param=floor(n/2);
        else
            param=min(n,str2double(law(5:(length(law)-1))));
        end

        R_v = ((n - param)/param)*ones(1,n);

    case 'Loo'
        R_v = (1/(n-1))*ones(1,n);

    case 'Rad'
        if length(law)==3
            param=1/2;
        else
            param=str2double(law(5:(length(law)-1)));
        end

        if (param<1 && param>0)
            R_v = ((1-param)/param)*ones(1,n);
        else
            R_v = NaN*ones(1,n);
        end

    otherwise
        R_v = NaN*ones(1,n);

end%switch (law(1:3))

function C=C_W(n,law)
% computes the constant C_W for a sample size n and a resampling distribution given by 'law'
% possible values for the string law:
% 'Efr'    : Efron(n) weights
% 'Efr(M)' : Efron(M) weights
% 'Poi'    : Poisson(1) weights
% 'Poi(L)' : Poisson(L) weights
% 'Rho'    : Rho(n/2) weights (or (n-1)/2 if n is odd)
% 'Rho(q)' : Rho(q) weights
% 'Loo'    : Rho(n-1) weights
% 'Rad'    : Rad(1/2) weights
% 'Rad(p)' : Rad(p) weights

switch (law(1:3))
    case 'Efr'
        if length(law)==3
            param=n;
        else
            param=str2double(law(5:(length(law)-1)));
        end

	C = param/n;

    case 'Poi'
        if length(law)==3
            param=1;
        else
            param=str2double(law(5:(length(law)-1)));
        end

	C = param;

    case 'Rho'
        if length(law)==3
            param=floor(n/2);
        else
            param=min(n,str2double(law(5:(length(law)-1))));
        end

	C = param/(n - param);

    case 'Loo'
	C = n-1;

    case 'Rad'
        if length(law)==3
            param=1/2;
        else
            param=str2double(law(5:(length(law)-1)));
        end

        if (param<1 && param>0)
            C = param/(1-param) ;
        else
            C=NaN;
        end

    otherwise
	C=NaN;

end%switch (law(1:3))


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%% Draw random weight vectors 

function W_mat=gener_Wmat(n,law,B)
% generates a matrix of random weights 'W_v' of size B*n with distribution given by 'law'
% possible values for the string law:
% 'Efr'    : Efron(n) weights
% 'Efr(M)' : Efron(M) weights
% 'Poi'    : Poisson(1) weights (requires the Statistics Toolbox)
% 'Poi(L)' : Poisson(L) weights (requires the Statistics Toolbox)
% 'Rho'    : Rho(n/2) weights (or (n-1)/2 if n is odd)
% 'Rho(q)' : Rho(q) weights
% 'Loo'    : Rho(n-1) weights
% 'Rad'    : Rad(1/2) weights
% 'Rad(p)' : Rad(p) weights

W_mat = zeros(B,n);

for i=1:B
	W_mat(i,:)=gener_W(n,law);
end

function W_v=gener_W(n,law)
% generates a random weight vector 'W_v' of size n with distribution given by 'law'
% possible values for the string law:
% 'Efr'    : Efron(n) weights
% 'Efr(M)' : Efron(M) weights
% 'Poi'    : Poisson(1) weights (requires the Statistics Toolbox)
% 'Poi(L)' : Poisson(L) weights (requires the Statistics Toolbox)
% 'Rho'    : Rho(n/2) weights (or (n-1)/2 if n is odd)
% 'Rho(q)' : Rho(q) weights
% 'Loo'    : Rho(n-1) weights
% 'Rad'    : Rad(1/2) weights
% 'Rad(p)' : Rad(p) weights

W_v = zeros(1,n);

switch law(1:3)
    case 'Efr'
        if length(law)==3
            param=n;
        else
            param=str2double(law(5:(length(law)-1)));
        end

        for i=1:param
            Uj = ceil(n*rand(1));
            W_v(Uj) = W_v(Uj) + (n/param);
        end

    case 'Poi'
	if length(law)==3
		param=1;
	else
		param=str2double(law(5:(length(law)-1)));
	end

	W_v=poissrnd(param*ones(1,n))/param;

    case 'Rho'
	if length(law)==3
		param=floor(n/2);
	else
		param=min(n,str2double(law(5:(length(law)-1))));
	end

	sig = randperm(n);
	W_v(sig(1:param))=(n/param)+zeros(1,param);

    case 'Loo'
	W_v = (n/(n-1))*ones(1,n);
	W_v(ceil(n*rand(1)))=0;

    case 'Rad'
        if length(law)==3
            param=1/2;
        else
        	param=str2double(law(5:(length(law)-1)));
        end

    if (param>0 && param<1)
        W_v=(1/param)*((rand(1,n)<param));
    else
        W_v=NaN*ones(1,n);
    end

    otherwise
	W_v=NaN*ones(1,n);

end%switch lawdeb


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%% Exact computation of the resampling penalty 

	
function [pen,Remp,z] = RP_exact(X_v,Y_v,model,R1W_v,R2W_v)
% This function computes the empirical risk and the exact resampling penalty associated to some histogram model
% WITHOUT multiplying the penalty by the constant C_W (this should be made after)
%
% INPUT :
% 'X_v' is the vector of the X_i
% 'Y_v' is the vector of the Y_i
% 'model' is the vector of the possible jump points in the model (including the bounds of X)
%         for instance model = [0 0.5 1] for the model of regular histograms of dimension 2
% 'R1W_v' is a vector of size n containing the value of R_{1,W}(n,phl) for n*phl=1, ..., n
% 'R2W_v' is a vector of size n containing the value of R_{2,W}(n,phl) for n*phl=1, ..., n
%		R1W_v and R2W_v can be computed with the functions R1W and R2W 
%
% OUTPUT :
% 'pen' is a real number, equal to the value of the resampling penalty
% 'Remp' is a real number, equal to the empirical (quadratic) risk of the least squares estimator
% 'z' is the empirical risk minimizer (represented by a vector)

d=numel(model)-1;% dimension of the model
n=numel(X_v);% number of data points

% First, compute the ERM
[z,Rempv,Rwempv,minnphl]=ERMhist(X_v,Y_v,ones(size(X_v)),model);
Remp=sum(Rempv);
if minnphl==0
	pen=Inf;
	z=NaN*ones(1,d);
else
	
pen_v = zeros(1,d);

for i=1:d
	Xind=(X_v >= model(i) & X_v < model(i+1));% points in the i-th interval
	nphl=sum(Xind);% number of points in the i-th interval
	if nphl<2
		pen_v(i)=NaN;
	else
		pen_v(i)=(R1W_v(nphl)+R2W_v(nphl))*(nphl*(sum((Y_v(Xind)).^2)) - (sum(Y_v(Xind)))^2)/(n*(nphl*(nphl - 1)));
	end%if nphl<2
end%for i=1:d

pen=sum(pen_v);

end%if minnphl==0

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%% Monte-Carlo approximation of the resampling penalty

function [pen,Remp,z] = RP_MonteCarlo(X_v,Y_v,model,W_mat)
% This function computes the empirical risk and the resampling penalty associated to some histogram model by Monte-Carlo approximation
% WITHOUT multiplying the penalty by the constant C_W (this should be made after)
%
% INPUT :
% 'X_v' is the vector of the X_i
% 'Y_v' is the vector of the Y_i
% 'model' is the vector of the possible jump points in the model (including the bounds of X)
%         for instance model = [0 0.5 1] for the model of regular histograms of dimension 2
% 'W_mat' is a B*n matrix whose rows are independent weight vectors
%	In principle, W_mat should have been generated with gener_Wmat
%
% OUTPUT :
% 'pen' is a real number, equal to the value of the resampling penalty
% 'Remp' is a real number, equal to the empirical (quadratic) risk of the least squares estimator
% 'z' is the empirical risk minimizer (represented by a vector)

d=numel(model)-1;% dimension of the model
B=size(W_mat,1);% number of independent weight vectors used to compute the penalty

% First, compute the ERM
[z,Rempv,Rwempv,minnphl]=ERMhist(X_v,Y_v,ones(size(X_v)),model);
Remp=sum(Rempv);
if minnphl==0
	pen=Inf;
	z=NaN*ones(1,d);
else
	
pen_mat = zeros(B,d);% matrix which will contain the contribution of each interval to the resampling penalty, separately for each weight vector. The idea is that some elements will be equal to NaN (when W_lambda = 0), so that they should not be taken into account for computing the resampling penalty

for i=1:B
	% Compute the ERM for each weighted sample
	[zz,Rempv,Rwempv,minnphl]=ERMhist(X_v,Y_v,W_mat(i,:),model);
	pen_mat(i,:)=Rempv-Rwempv;
end%for i=1:B


% The contribution of the i-th interval to the penalty is computing by averaging over the weights for which the weighted ERM is well-defined on the i-th interval
pen_v=zeros(1,d);

for j=1:d
	tmp=pen_mat(:,j);
	try 
		pen_v(j)=mean(tmp(isfinite(tmp)));
    catch
		pen_v(j)=Inf;
	end%if sum(isfinite(tmp))>0
end%for j=1:d

% The final penalty is the sum of the contributions of all the intervals

pen=sum(pen_v);

end%if minnphl==0


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%% weighted least squares estimator for histogram models


function [z,Rempv,Rwempv,minnphlW]=ERMhist(X_v,Y_v,W_v,model)
% This function computes the weighted least-squares estimator associated to some histogram model
% INPUT :
% 'X_v' is the vector of the X_i
% 'Y_v' is the vector of the Y_i
% 'W_v' is a vector of weights to be assigned to the data
% 'model' is the vector of the possible jump points in the model (including the bounds of X)
%         for instance model = [0 0.5 1] for the model of regular histograms of dimension 2
%
% OUTPUT :
% 'z' is the vector of the same size as model, where z(i) is the value of the weighted least squares estimator on the i-th piece of the partition
% 'Remp' is a real number, equal to the empirical (quadratic) risk of the weighted least squares estimator
% 'Rwemp' is a real number, equal to the weighted empirical (quadratic) risk of the weighted least squares estimator
% 'minnphlW' is the minimal number of X_i in one piece of the partition

d=numel(model)-1;% dimension of the model
n=numel(X_v);% number of data points

z=zeros(1,d);
Rempv=zeros(1,d);
Rwempv=zeros(1,d);
minnphlW=n;

for i=1:d
	Xind=(X_v >= model(i) & X_v < model(i+1));% points in the i-th interval
	nphl=sum(Xind);% number of points in the i-th interval
	nphlW=sum(W_v(Xind));% total weight of the points in the i-th interval

	if nphlW>0
		z(i)=sum((Y_v(Xind)).*(W_v(Xind)))/nphlW;% value zeros(d,2);of the weighted ERM on the i-th interval
		Rempv(i)=sum(((Y_v(Xind)) - (z(i)*ones(1,nphl))).^2)/n;% contribution of the i-th interval to the empirical risk of the weighted ERM
		Rwempv(i)=sum(W_v(Xind).*(((Y_v(Xind)) - (z(i)*ones(1,nphl))).^2))/n;% contribution of the i-th interval to the empirical risk of the weighted ERM
	else%if nphlW=0
		z(i)=NaN;
		Rempv(i)=NaN;
		Rwempv(i)=NaN;
	end%if nphlW>0
	minnphlW=min(minnphlW,nphlW);% update of minnphl
end%for i=1:size(model,2)







